package loquebot.drives;

import java.util.ArrayList;
import java.util.Iterator;

import cz.cuni.pogamut.MessageObjects.AddWeapon;
import cz.cuni.pogamut.MessageObjects.Item;
import cz.cuni.pogamut.MessageObjects.Weapon;
import cz.cuni.pogamut.MessageObjects.Ammo;

import loquebot.Main;
import loquebot.util.LoqueWeaponInfo;
import loquebot.memory.LoqueMemory;

/**
 * Necessary support for handling power-up drive. This class implements all
 * composite methods and decission making routines used by the logic in
 * {@link LoquePowerUp} to make {@link LoquePowerUp} look more simple and
 * understandalbe on laic glance.
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public abstract class LoquePowerUpBase extends LoqueDrive
{
    /**
     * Retreives list of nearby weapons that could be easily picked up.
     * @return List of pickable items.
     */
    protected ArrayList<Item> getListOfNearbyWeapons ()
    {
        // get all visible and useful items
        ArrayList<Item> items = memory.items.getVisibleItems(true, false);

        // now, filter the list..
        Iterator<Item> i = items.iterator ();
        while (i.hasNext ())
        {
            Item item = i.next ();

            // not reachable?
            if (!item.reachable)
            {
                i.remove ();
                continue;
            }
            // too far away?
            if (memory.self.getSpaceDistance(item.location) > 500)
            {
                i.remove ();
                continue;
            }
        }
        // return filtered list
        return items;
    }

    /*========================================================================*/

    /**
     * Filters lists of given weapons and ammos according to target score.
     *
     * @param weapons List of weapons to choose from.
     * @param ammos List of ammos to choose from.
     * @param chosen List to be cleared and re-populated.
     * @param score Target weapon score.
     * @return Size of the populated list.
     */
    private int filterItems (ArrayList<Weapon> weapons, ArrayList<Ammo> ammos, ArrayList<Item> chosen, int score)
    {
        // retreive agent's skill level
        int skill = body.initializer.getBotSkillLevel ();

        // clear the destination list
        chosen.clear ();

        // run through the list of weapons
        for (Weapon weapon : weapons)
            // check the target score of the weapon
            if (LoqueWeaponInfo.getInfo(weapon.weaponType).generalScore(skill) >= score)
                // add it to the list
                chosen.add(weapon);

        // run through the list of ammos
        for (Ammo ammo : ammos)
            // check the target score of the ammo
            if (LoqueWeaponInfo.getInfo(ammo.typeOfWeapon).generalScore(skill) >= score)
                // add it to the list
                chosen.add(ammo);

        return chosen.size ();
    }

    /**
     * Filters lists of weapons and ammos to achieve the best target score.
     * @param weapons List of weapons to choose from.
     * @param ammos List of ammos to choose from.
     * @param min Minimum count of chosen items.
     * @return List of chosen items.
     */
    private ArrayList<Item> filterBestItems (ArrayList<Weapon> weapons, ArrayList<Ammo> ammos, int min)
    {
        ArrayList<Item> chosen = new ArrayList<Item> (weapons.size () + ammos.size ());

        // get list of the best weapons.. is it enough?
        if (filterItems (weapons, ammos, chosen, 780) >= min)
            return chosen;

        // get list of medium weapons.. is it enough?
        if (filterItems (weapons, ammos, chosen, 600) >= min)
            return chosen;

        // get list of useable weapons.. is it enough?
        if (filterItems (weapons, ammos, chosen, 400) >= min)
            return chosen;

        // well, return any weapons available..
        chosen.clear ();
        chosen.addAll(weapons);
        chosen.addAll(ammos);
        return chosen;
    }

    /*========================================================================*/

    /**
     * Retreives list of the best useful weapons available.
     *
     * @return List of the chosen weapon pickups.
     */
    protected ArrayList<Item> getListOfGoodWeapons ()
    {
        // retreive at least two weapons with the highest score, no ammos
        return filterBestItems (
            memory.items.getAllWeapons (true, false),
            new ArrayList<Ammo> (),
            2
        );
    }

    /*========================================================================*/

    /**
     * Retreives list of the best useful ammo and weapons available.
     *
     * @return List of the chosen ammo pickups.
     */
    protected ArrayList<Item> getListOfGoodItems ()
    {
        // retreive at least two/six weapons/ammos with the highest score
        return filterBestItems (
            memory.items.getAllWeapons (true, false),
            memory.items.getAllAmmos (true, false),
            1 + (int)(Math.random () * 3)
        );
    }

    /*========================================================================*/

    /**
     * Retreives list of useful available healths.
     *
     * @return List of the chosen health pickups.
     */
    protected ArrayList<Item> getListOfGoodHealth ()
    {
        // retreive list of useful healths on the map
        @SuppressWarnings ("unchecked")
        ArrayList<Item> items = (ArrayList) memory.items.getAllHealths (true, false);

        return items;
    }

    /*========================================================================*/

    /**
     * Retreives list of useful available armors.
     *
     * @return List of the chosen armor pickups.
     */
    protected ArrayList<Item> getListOfGoodArmor ()
    {
        // retreive list of useful armors on the map
        @SuppressWarnings ("unchecked")
        ArrayList<Item> items = (ArrayList) memory.items.getAllArmors (true, false);

        return items;
    }

    /*========================================================================*/

    /**
     * Retreives list of useful available specials.
     *
     * @return List of the chosen special pickups.
     */
    protected ArrayList<Item> getListOfGoodSpecial ()
    {
        // retreive list of useful specials on the map
        @SuppressWarnings ("unchecked")
        ArrayList<Item> items = (ArrayList) memory.items.getAllSpecials (true, false);

        return items;
    }

    /*========================================================================*/

    /**
     * Retreives maximum weapon score that could be achieved with the best
     * weapon currently in the inventory.
     *
     * <p>Different weapons give different values. This can be used to determine
     * potential powered-up status.</p>
     *
     * <p>Note: The score is not based on how much ammo the agent has for each
     * weapon. On the contrary, the score immitates weapons with full ammo.</p>
     *
     * @return Best weapon score in the inventory.
     */
    protected int getTopInventoryScore ()
    {
        int weaponScore = 0;

        // search the list of current weapons
        for (AddWeapon weapon: memory.inventory.getWeapons ())
        {
            // retreive and add weapon score of each weapon
            weaponScore = Math.max (
                weaponScore,
                LoqueWeaponInfo.getInfo(weapon.weaponType).inventoryScore(
                    body.initializer.getBotSkillLevel (), weapon.maxAmmo
                )
            );
        }

        return weaponScore;
    }

    /**
     * Retreives total approximate weapon score of the current inventory.
     *
     * <p>Different weapons give different values. The score is also based on
     * how much ammo we have for each weapon. This can be used to determine the
     * overall powered-up status.</p>
     *
     * @return Current weaponry score of the inventory.
     */
    protected int getTotalInventoryScore ()
    {
        int weaponScore = 0;

        // search the list of current weapons
        for (AddWeapon weapon: memory.inventory.getWeapons ())
        {
            // retreive and add weapon score of each weapon
            weaponScore += LoqueWeaponInfo.getInfo(weapon.weaponType).inventoryScore(
                body.initializer.getBotSkillLevel (), weapon.currentAmmo
            );
        }

        return weaponScore;
    }

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoquePowerUpBase (Main main, LoqueMemory memory)
    {
        super (main, memory);
    }
}